/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: Config_netX50.c $
   Last Modification:
    $Author: Robert $
    $Modtime: 26.02.08 17:04 $
    $Revision: 12216 $

   Targets:
    rcX          : yes

   Description:

    Configuration file for netX 50 based hardware

      * HW included:
        - UART0 + IRQ
        - TIMER0 + IRQ
        - SPI + IRQ
        - SYSLED

   Changes:

     Version  Date           Author  Description
     ----------------------------------------------------------------------------------
      1       01.03.2011     SS      initial release for netX50
**************************************************************************************/

/*
************************************************************
*   Inclusion Area
************************************************************
*/

/* Standard Includes */
#include "rX_Includes.h" 
#include "mmio_config.h"
#include "boottokens.h"

#include "ToolkitSample.h"

/*
************************************************************
*   Externals needed for patches
************************************************************
*/

/*******************************************************************************
  Local Defines - Memory size configuration
*******************************************************************************/
/* netX CPU peripherals clock rate */
#define NETX_FREQUENCY_100MHZ          100000000

#define NXHX_SPI_CHANNEL 1 /* Channel 0 on connector X5
                              Channel 1 on connector X6 */
                              
/*******************************************************************************
  Definition of the Hardware Timer Instances
*******************************************************************************/
STATIC CONST FAR RX_HWTIMER_SET_T atrXHwTim[] = {
  {
    {"SYSTIMER",RX_PERIPHERAL_TYPE_TIMER,0}, /* System Timer */
    0, /* use GPIO_counter0 */
    1000, /* 1000 microseconds = 1msec */
    TRUE, /* Continuous Mode */
    TRUE, /* Interrupt enabled */
    FALSE, /* No external Clock */
    RX_HWTIMER_TRIGGER_NONE, /* No Trigger */
    0, /* No I/O reference */
    0 /* No Prescaler */
  }
};

/*******************************************************************************
  Configuration of the SPI Instance
*******************************************************************************/
STATIC CONST FAR RX_SPISLAVE_SET_T atrXSpi[] = {
  {
    {"SERDPM", RX_PERIPHERAL_TYPE_SPI, 0}, // first SPI instance
    1,                                     // index of the slave's spi port
    1,                                     // index of the slave's chip select
    RX_SPI_MODE3,                          // spi mode for this slave device
    RX_SPI_SPEED_5_0MHz,                   // clock speed for this slave
    100,                                   // size of a burst block
    0                                      // delay between 2 burst blocks
  },
};

/*******************************************************************************
  Configuration of the UART Instance
*******************************************************************************/
STATIC CONST FAR RX_UART_SET_T atrXUrt[] = {
  {
    {"REALUART", RX_PERIPHERAL_TYPE_UART, 0},           // first "REALUART" instance
    0,                                                  // use UART 0 (first real UART)
    RX_UART_BAUDRATE_115200,                            // baud rate 115,2k
    RX_UART_PARITY_NONE,                                // no parity
    RX_UART_STOPBIT_1,                                  // 1 stop bit
    RX_UART_DATABIT_8,                                  // 8 data bits
    0,                                                  // "rx ready" trigger level for rx FIFO (set to 1..16 to enable FIFO)
    0,                                                  // "tx empty" trigger level for tx FIFO (set to 1..16 to enable FIFO)
    RX_UART_RTS_NONE,                                   // no RTS in use
    RX_UART_RTS_DEFAULT,                                // no RTS in use
    0,                                                  // no RTS forerun
    0,                                                  // no RTS trail
    RX_UART_CTS_NONE,                                   // no CTS in use
    RX_UART_CTS_DEFAULT                                 // default CTS handling
  },
};

/*******************************************************************************
  Definition of the Interrupt-Instances
*******************************************************************************/
STATIC CONST FAR RX_INTERRUPT_SET_T atrXInt[] = {
  {{"SYSTIMER",RX_PERIPHERAL_TYPE_INTERRUPT,0}, /* System Timer interrupt */
    SRT_NX10_vic_irq_status_timer0, /* Use external Timer0 Interrupt */
    23, /* Priority 29 */
    RX_INTERRUPT_MODE_SYSTEM, /* Allow interrupt to be a thread */
    RX_INTERRUPT_EOI_AUTO, /* EOI self by RX */
    RX_INTERRUPT_TRIGGER_RISING_EDGE, /* Edge triggered */
    RX_INTERRUPT_PRIORITY_STANDARD, /* Normal Priority */
    RX_INTERRUPT_REENTRANCY_DISABLED, /* Interrupt itself is reentrant */
  },
  /* UART port 0 */
  {
    {"UART",RX_PERIPHERAL_TYPE_INTERRUPT,0}, /* Verbose UART interrupt */
    SRT_NX10_vic_irq_status_uart0, /* Use external UART0 Interrupt */
    27, /* Priority 28 */
    RX_INTERRUPT_MODE_TASK, /* Interrupt will be handled in interrupt mode */
    RX_INTERRUPT_EOI_AUTO, /* EOI automatically by RX */
    RX_INTERRUPT_TRIGGER_RISING_EDGE, /* Edge triggered */
    RX_INTERRUPT_PRIORITY_STANDARD, /* Normal Priority */
    RX_INTERRUPT_REENTRANCY_DISABLED, /* Interrupt itself is not reentrant */
    TSK_PRIO_7,
    TSK_TOK_7,
    1024
  },
  {
    {"SPI", RX_PERIPHERAL_TYPE_INTERRUPT, 0},     // SPI
    SRT_NX10_vic_irq_status_spi,                  // use SPI interrupt bit in the VIC IRQ status register
    25,                                           // interrupt priority
    RX_INTERRUPT_MODE_TASK,                       // create a separate task for the interrupt
    RX_INTERRUPT_EOI_AUTO,                        // EOI handled automatically
    RX_INTERRUPT_TRIGGER_RISING_EDGE,             // edge triggered
    RX_INTERRUPT_PRIORITY_STANDARD,               // normal priority
    RX_INTERRUPT_REENTRANCY_DISABLED,             // interrupt is not reentrant
    TSK_PRIO_8,                                   // interrupt task priority
    TSK_TOK_8,                                    // interrupt task token
    1024                                          // interrupt task stack size
  },
};

/*******************************************************************************
  Configuration of the LED Instances
*******************************************************************************/
STATIC CONST FAR RX_LED_SET_T atrXLed[] = {
  { // SYS yellow
    {"READY", RX_PERIPHERAL_TYPE_LED, 0},  // Peripheral Header
    {"RDYRUN", RX_PERIPHERAL_TYPE_LED, 0}, // Register Destination Definition
    {0},                                   // Optional Register to make LED Pin to output at startup
    {0},                                   // Optional Register to make LED Pin to output at startup
    {RX_LED_VALUE_TYPE_OR, 1, 0x01},       // LED Enable Register and Value
    {RX_LED_VALUE_TYPE_OR, 1, 0x01},       // LED Disable Register and Value
    NULL                                   // Set of LED functions
  },
  { // SYS green
    {"RUN", RX_PERIPHERAL_TYPE_LED, 0},    // Peripheral Header
    {"RDYRUN", RX_PERIPHERAL_TYPE_LED, 0}, // Register Destination Definition
    {0},                                   // Optional Register to make LED Pin to output at startup
    {0},                                   // Optional Register to make LED Pin to output at startup
    {RX_LED_VALUE_TYPE_OR, 2, 0x02},       // LED Enable Register and Value
    {RX_LED_VALUE_TYPE_OR, 2, 0x02},       // LED Disable Register and Value
    NULL                                   // Set of LED functions
  },
};

/*******************************************************************************
  Peripheral Configuration Summary to handed over to the RX-Kernel
*******************************************************************************/
STATIC CONST FAR RX_PERIPHERAL_CONFIG_T atrXCfg[] = 
{
  {RX_PERIPHERAL_TYPE_TIMER,atrXHwTim,MAX_CNT(atrXHwTim)},
  {RX_PERIPHERAL_TYPE_INTERRUPT,atrXInt,MAX_CNT(atrXInt)},
};

/*******************************************************************************
  Static Task List
*******************************************************************************/
STATIC CONST FAR RX_STATIC_TASK_T FAR atrXStaticTasks[] = 
{
  {"ToolkitExample",                      /* Set Identification */
   TSK_PRIO_36, TSK_TOK_36,               /* Set Priority,and Token ID */
   0,                                     /* Set Instance to 0 */
   0,                                     /* Pointer to Stack */
   0x1000,                                /* Size of Task Stack */
   0,                                     /* Threshold to maximum possible value */
   RX_TASK_AUTO_START,                    /* Start task automatically */
   ToolkitSampleTask,                     /* Task function to schedule */
   NULL,                                  /* Function called when Task will be deleted */
   0,                                     /* Startup Parameter */
   {0,0,0,0,0,0,0,0}                      /* Reserved Region */
  },
};

/*******************************************************************************
  Peripheral Driver Configuration loaded after the Kernel has been started
*******************************************************************************/
STATIC CONST FAR RX_DRIVER_PERIPHERAL_CONFIG_T atrXDrvCfgPost[] = {
  {DrvLedInit,RX_PERIPHERAL_TYPE_LED,atrXLed,MAX_CNT(atrXLed)},
  {DrvUrtInit, RX_PERIPHERAL_TYPE_UART, atrXUrt, MAX_CNT(atrXUrt)},
  {DrvSpiInit, RX_PERIPHERAL_TYPE_SPI, atrXSpi, MAX_CNT(atrXSpi)}
};

/*******************************************************************************
  Common Parameter block for rX_SysEnterKernelExt
*******************************************************************************/
RX_ENTERKERNEL_PARAM_T trXEnterKernelParam=
{
    /* CPU clock rate */
    NETX_FREQUENCY_100MHZ,

    /* Timer interrupt task priority */
    {TSK_PRIO_DEF_RX_TIMER, 1024},

    /* Pointer to static Task-List */
    {atrXStaticTasks, MAX_CNT(atrXStaticTasks)},

    /* Pointer to rx kernel modules list */
    {0, 0},

    /* Pointer to the Peripherals-List */
    {atrXCfg, MAX_CNT(atrXCfg)},

    /* Pointer to the Post Peripherals-List / LoadDrivers included into */
    {atrXDrvCfgPost, MAX_CNT(atrXDrvCfgPost)},

    /* Pointer to optional Jump Table */
    {NULL, 0},

    /* Callback for special initialization */
    NULL,

    /* Pointer to the Middleware List */
    {0, 0},

    /* Scheduler component (if another scheduler is desired) */
    (RX_SCHEDULER_FUNCTIONS_T*)&g_tMLQueueScheduler,

    /* Cache enable flags */
    {FALSE, FALSE},
    {FALSE},
    NULL,
    {0},
};

/*******************************************************************************
  Configuration of the MMIO matrix
*******************************************************************************/
STATIC CONST MMIO_CONFIG_T s_atMmioConfig[] =
{
#if NXHX_SPI_CHANNEL == 0 /* X5 */
  {  0, MMIO_CONFIG_SPI1_MOSI,           0, 0 },
  {  1, MMIO_CONFIG_SPI1_MISO,           0, 0 },
  {  2, MMIO_CONFIG_SPI1_CLK,            0, 0 },
  {  3, MMIO_CONFIG_SPI1_CS1N,           0, 0 }, 
  {  4, MMIO_CONFIG_PIO,                 0, 0 },                                     
  {  6, MMIO_CONFIG_PIO,                 0, 0 },                                     
  {  5, MMIO_CONFIG_PIO,                 0, 0 },  
  {  7, MMIO_CONFIG_PIO,                 0, 0 },                                     
  {  9, MMIO_CONFIG_SPI1_CS1N,           0, 0 },   
  { 19, MMIO_CONFIG_PIO,                 0, 0 }, 
#elif NXHX_SPI_CHANNEL == 1 /* X6 */
  {  0, MMIO_CONFIG_PIO,                 0, 0 },
  {  1, MMIO_CONFIG_PIO,                 0, 0 },
  {  2, MMIO_CONFIG_PIO,                 0, 0 },
  {  3, MMIO_CONFIG_PIO,                 0, 0 }, 
  {  4, MMIO_CONFIG_SPI1_MISO,           0, 0 },                                     
  {  6, MMIO_CONFIG_SPI1_CS1N,           0, 0 },                                     
  {  5, MMIO_CONFIG_SPI1_MOSI,           0, 0 },  
  {  7, MMIO_CONFIG_SPI1_CLK,            0, 0 },                                     
  {  9, MMIO_CONFIG_PIO,                 0, 0 },   
  { 19, MMIO_CONFIG_SPI1_CS1N,           0, 0 }, 
#endif                                  
  { 20, MMIO_CONFIG_UART0_RXD,           0, 0 },                
  { 21, MMIO_CONFIG_UART0_TXD,           0, 0 },                                        
};
 
/*  =====================================================================================

  Function: main

            - Enters the RX-kernel mode forever

  ---------------------------------------------------------------------------------------

  Input   : -

  Output  : -

  Return  : -


  ======================================================================================= */
INT main (void)
{
  volatile RX_FATAL erXFat; /* Fatal Error value, in case function would return */
  int iIdx = 0;
         
  /* Configure MMIO pin multiplexer */
  for(iIdx = 0; iIdx < (int)(sizeof(s_atMmioConfig) / sizeof(s_atMmioConfig[0])); iIdx++)
  {
    volatile unsigned long* pulMmioConfig = (volatile unsigned long*)NX10_NETX_MMIO_CTRL_AREA;
    volatile unsigned long* pulAccessKey  = (volatile unsigned long*)Adr_NX10_asic_ctrl_access_key;
    
    // Unlock MMIO Config register
    *pulAccessKey = *pulAccessKey;
    
    pulMmioConfig[s_atMmioConfig[iIdx].bMmio] = s_atMmioConfig[iIdx].bFunction << SRT_NX10_mmio0_cfg_mmio_sel             |
                                                ((s_atMmioConfig[iIdx].fInvertIn)  ? MSK_NX10_mmio0_cfg_mmio_in_inv  : 0) |
                                                ((s_atMmioConfig[iIdx].fInvertOut) ? MSK_NX10_mmio0_cfg_mmio_out_inv : 0);
  }
         
  /* Boot the Kernel by using the set Call-Pointer and hand over the static Task-List and the Peripherals-List */
  erXFat = rX_SysEnterKernelExt(&trXEnterKernelParam);

  /* We should never return to here */  
  while(erXFat!=RX_OK);
  
  return(0); /* Prevent Compiler warning because of non-void function */
}
